<?php

namespace App\Http\Controllers;

use Modules\Academic\Models\Staff;
use Modules\Academic\Models\Subject;
use Modules\Academic\Models\Teacher;
use Modules\Academic\Models\Signature;
use Modules\Examination\Models\ClassExam;
use Modules\Academic\Services\StudentService;
use Modules\SystemConfiguration\Models\Setting;

class WebsiteController extends Controller
{
    public function __construct(
        private readonly StudentService $studentService,
    ) {}

    public function index()
    {
        return view('welcome');
    }

    public function vital(string $encodedData)
    {
        try {
            // Decode
            $decoded = base64_decode($encodedData, true);

            if ($decoded === false) {
                return abort(404, 'Invalid request.');
            }

            $data = json_decode($decoded, true);

            // Validate structure
            if (
                !is_array($data) ||
                !isset($data['institute_id'], $data['branch_id'], $data['student_id']) ||
                !ctype_digit((string)$data['institute_id']) ||
                !ctype_digit((string)$data['branch_id']) ||
                !ctype_digit((string)$data['student_id'])
            ) {
                return abort(404, 'Invalid data format.');
            }

            $instituteId = (int) $data['institute_id'];
            $branchId    = (int) $data['branch_id'];
            $studentId   = (int) $data['student_id'];

            // Lookup student (with institute + branch if required)
            $student = $this->studentService->findStudentById($studentId);

            if (!$student) {
                return abort(404, 'Student not found.');
            }

            $exams = ClassExam::with('exam')
                ->where('class_id', $student->studentSession?->class_id)
                ->get()
                ->pluck('exam');

            $subjects = Subject::where('class_id', $student->studentSession?->class_id)->get();

            return view('vital.index', compact('student', 'exams', 'subjects', 'studentId', 'instituteId', 'branchId'));
        } catch (\Throwable $e) {
            return abort(404, 'Something went wrong.');
        }
    }

    public function studentIdCardList(string $encodedData)
    {

        try {
            // Decode Base64
            $decoded = base64_decode($encodedData, true);

            if ($decoded === false) {
                return abort(404, 'Invalid request.');
            }

            $data = json_decode($decoded, true);

            if (!is_array($data)) {
                return abort(404, 'Invalid data format.');
            }

            // Extract with defaults
            $class_id     = (int) ($data['class_id'] ?? 0);
            $section_id   = (int) ($data['section_id'] ?? 0);
            $group_id     = (int) ($data['group_id'] ?? 0);
            $validity_date = $data['validity'] ?? null;
            $instituteId  = (int) ($data['institute_id'] ?? 1);
            $branchId     = (int) ($data['branch_id'] ?? 1);

            // Fetch students
            $students = $this->studentService->getStudentsByClassSectionGroup(
                $class_id,
                $section_id,
                $group_id,
                200,
                $instituteId,
                $branchId
            );


            // Fetch settings
            $settings = Setting::where('institute_id', $instituteId)
                ->where('branch_id', $branchId)
                ->pluck('value', 'name')
                ->toArray();

            $principalSignature = Signature::first();

            return view('backend.students.id-card.card-list', compact(
                'students',
                'validity_date',
                'principalSignature',
                'instituteId',
                'branchId',
                'settings'
            ));
        } catch (\Throwable $e) {
            \Log::error('ID Card List error: ' . $e->getMessage());
            return abort(404, 'Something went wrong.');
        }
    }

    public function teacherIdCardList(string $encodedData)
    {
        try {
            // Decode Base64
            $decoded = base64_decode($encodedData, true);

            if ($decoded === false) {
                return abort(404, 'Invalid request.');
            }

            $data = json_decode($decoded, true);

            if (!is_array($data)) {
                return abort(404, 'Invalid data format.');
            }

            // Extract IDs
            $instituteId = (int) ($data['institute_id'] ?? 1);
            $branchId    = (int) ($data['branch_id'] ?? 1);

            // Fetch teachers
            $teachers = Teacher::select(
                '*',
                'teachers.id AS id',
                'teachers.status AS teacher_status',
            )
                ->where('teachers.institute_id', $instituteId)
                ->where('teachers.branch_id', $branchId)
                ->join('users', 'users.id', '=', 'teachers.user_id')
                ->orderBy('teachers.id', 'DESC')
                ->get();

            // Settings & signature
            $principalSignature = Signature::first();
            $settings = Setting::where('institute_id', $instituteId)
                ->where('branch_id', $branchId)
                ->pluck('value', 'name')
                ->toArray();

            return view('backend.teachers.id-card.card-list', compact(
                'teachers',
                'principalSignature',
                'instituteId',
                'branchId',
                'settings'
            ));
        } catch (\Throwable $e) {
            \Log::error('Teacher ID Card List error: ' . $e->getMessage());
            return abort(404, 'Something went wrong.');
        }
    }

    public function staffIdCardList(string $encodedData)
    {
        try {
            // Decode Base64
            $decoded = base64_decode($encodedData, true);

            if ($decoded === false) {
                return abort(404, 'Invalid request.');
            }

            $data = json_decode($decoded, true);

            if (!is_array($data)) {
                return abort(404, 'Invalid data format.');
            }

            // Extract IDs
            $instituteId = (int) ($data['institute_id'] ?? 1);
            $branchId    = (int) ($data['branch_id'] ?? 1);

            // Fetch staffs
            $staffs = Staff::select(
                '*',
                'staffs.id AS id',
            )
                ->where('staffs.institute_id', $instituteId)
                ->where('staffs.branch_id', $branchId)
                ->join('users', 'users.id', '=', 'staffs.user_id')
                ->get();

            // Settings & signature
            $principalSignature = Signature::first();
            $settings = Setting::where('institute_id', $instituteId)
                ->where('branch_id', $branchId)
                ->pluck('value', 'name')
                ->toArray();

            return view('backend.staffs.id-card.card-list', compact(
                'staffs',
                'principalSignature',
                'instituteId',
                'branchId',
                'settings'
            ));
        } catch (\Throwable $e) {
            \Log::error('Staff ID Card List error: ' . $e->getMessage());
            return abort(404, 'Something went wrong.');
        }
    }
}
