<?php

namespace Modules\Frontend\Repositories;

use Exception;
use App\Abstracts\EntityRepository;
use Illuminate\Database\Query\Builder;
use Modules\Frontend\Models\ReadyToJoinUs;
use Illuminate\Contracts\Pagination\Paginator;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Database\Eloquent\Builder as EloquentBuilder;

class ReadyToJoinUsRepository extends EntityRepository
{
    public string $table = ReadyToJoinUs::TABLE_NAME;

    protected array $fillableColumns = [
        'institute_id',
        'branch_id',
        'title',
        'description',
        'icon',
        'created_at',
        'updated_at',
    ];

    protected function getQuery(): Builder
    {
        return parent::getQuery();
    }

    public function getAll(array $filterData = []): Paginator
    {
        $filter = $this->getFilterData($filterData);
        $query = $this->getReadyToJoinUsQuery();

        if (! $filter['with_deleted']) {
            $query->whereNull("{$this->table}.deleted_at");
        }

        if (isset($filter['search']) && strlen($filter['search']) > 0) {
            $query = $this->filterSearchQuery($query, $filter['search']);
        }

        return $query
            ->orderBy($filter['orderBy'], $filter['order'])
            ->paginate($filter['perPage']);
    }

    protected function getFilterData(array $filterData = []): array
    {
        $defaultArgs = [
            'perPage' => 10,
            'search' => '',
            'orderBy' => 'id',
            'order' => 'desc',
            'with_deleted' => false,
        ];

        return array_merge($defaultArgs, $filterData);
    }

    private function getReadyToJoinUsQuery(): Builder
    {
        return $this->getQuery()
            ->select('*');
    }

    protected function filterSearchQuery(Builder|EloquentBuilder $query, string $searchedText): Builder
    {
        $searchable = "%$searchedText%";

        return $query->where('ready_to_join_us.title', 'LIKE', $searchable)
            ->orWhere('ready_to_join_us.description', 'LIKE', $searchable);
    }

    /**
     * @throws Exception
     */
    public function getByColumn(string $columnName, $columnValue, array $selects = ['*']): ?object
    {
        $user = $this->getReadyToJoinUsQuery()
            ->where($columnName, $columnValue)
            ->first();

        if (empty($user)) {
            throw new Exception(
                $this->getExceptionMessage(static::MESSAGE_ITEM_DOES_NOT_EXIST_MESSAGE),
                Response::HTTP_NOT_FOUND
            );
        }

        return $user;
    }

    public function getCount(array $filterData = []): int
    {
        $filter = $this->getFilterData($filterData);

        $query = $this->getQuery();

        if (! $filter['with_deleted']) {
            $query->whereNull("{$this->table}.deleted_at");
        }

        return $query->count();
    }

    /**
     * @throws Exception
     */
    public function create(array $data): object
    {
        try {
            $data = $this->prepareForDB($data);
            $userId = $this->getQuery()->insertGetId($data);
            $user = ReadyToJoinUs::find($userId);

            return $user;
        } catch (Exception $exception) {
            throw new Exception($exception->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * @throws Exception
     */
    public function prepareForDB(array $data, ?object $item = null): array
    {
        $data = parent::prepareForDB($data, $item);
        if (empty($item)) {
            $data['created_at'] = now();
            $data['institute_id'] = $this->getCurrentInstituteId();
            $data['branch_id'] = $this->getCurrentBranchId();
            if (! empty($data['icon']) && $data['icon'] instanceof \Illuminate\Http\UploadedFile) {
                $data['icon'] = fileUploader('ready_to_join_us/', 'png', $data['icon']);
            }
        } else {
            if (! empty($data['icon']) && $data['icon'] instanceof \Illuminate\Http\UploadedFile) {
                $data['icon'] = fileUploader('ready_to_join_us/', 'png', $data['icon'], $item->icon);
            }
            $data['updated_at'] = now();
        }

        return $data;
    }

    /**
     * @throws Exception
     */
    public function update(int $id, array $data): ?object
    {
        try {
            $user = ReadyToJoinUs::find($id);
            $data = $this->prepareForDB($data, $user);
            parent::update($id, $data);

            return $this->getById($id);
        } catch (Exception $exception) {
            throw new Exception($exception->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    protected function getExceptionMessages(): array
    {
        $exceptionMessages = parent::getExceptionMessages();

        $userExceptionMessages = [
            static::MESSAGE_ITEM_DOES_NOT_EXIST_MESSAGE => 'Ready To Join Us does not exist.',
            static::MESSAGE_ITEM_COULD_NOT_BE_DELETED => 'Ready To Join Us could not be deleted.',
        ];

        return array_merge($exceptionMessages, $userExceptionMessages);
    }
}
