<?php

namespace Modules\Frontend\Repositories;

use Exception;
use App\Abstracts\EntityRepository;
use Illuminate\Database\Query\Builder;
use Modules\Frontend\Models\MobileAppSection;
use Illuminate\Contracts\Pagination\Paginator;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Database\Eloquent\Builder as EloquentBuilder;

class MobileAppSectionRepository extends EntityRepository
{
    public string $table = MobileAppSection::TABLE_NAME;

    protected array $fillableColumns = [
        'institute_id',
        'branch_id',
        'title',
        'heading',
        'description',
        'image',
        'feature_one',
        'feature_two',
        'feature_three',
        'play_store_link',
        'app_store_link',
        'created_at',
        'updated_at',
    ];

    protected function getQuery(): Builder
    {
        return parent::getQuery();
    }

    public function getAll(array $filterData = []): Paginator
    {
        $filter = $this->getFilterData($filterData);
        $query = $this->getMobileAppSectionQuery();

        if (! $filter['with_deleted']) {
            $query->whereNull("{$this->table}.deleted_at");
        }

        if (isset($filter['search']) && strlen($filter['search']) > 0) {
            $query = $this->filterSearchQuery($query, $filter['search']);
        }

        return $query
            ->orderBy($filter['orderBy'], $filter['order'])
            ->paginate($filter['perPage']);
    }

    protected function getFilterData(array $filterData = []): array
    {
        $defaultArgs = [
            'perPage' => 10,
            'search' => '',
            'orderBy' => 'id',
            'order' => 'desc',
            'with_deleted' => false,
        ];

        return array_merge($defaultArgs, $filterData);
    }

    private function getMobileAppSectionQuery(): Builder
    {
        return $this->getQuery()
            ->select(
                'mobile_app_sections.id',
                'mobile_app_sections.institute_id',
                'mobile_app_sections.title',
                'mobile_app_sections.heading',
                'mobile_app_sections.description',
                'mobile_app_sections.image',
                'mobile_app_sections.feature_one',
                'mobile_app_sections.feature_two',
                'mobile_app_sections.feature_three',
                'mobile_app_sections.play_store_link',
                'mobile_app_sections.app_store_link',
                'mobile_app_sections.created_at',
                'mobile_app_sections.updated_at',
            );
    }

    protected function filterSearchQuery(Builder|EloquentBuilder $query, string $searchedText): Builder
    {
        $searchable = "%$searchedText%";

        return $query->where('mobile_app_sections.title', 'LIKE', $searchable)
            ->orWhere('mobile_app_sections.heading', 'LIKE', $searchable);
    }

    /**
     * @throws Exception
     */
    public function getByColumn(string $columnName, $columnValue, array $selects = ['*']): ?object
    {
        $user = $this->getMobileAppSectionQuery()
            ->where($columnName, $columnValue)
            ->first();

        if (empty($user)) {
            throw new Exception(
                $this->getExceptionMessage(static::MESSAGE_ITEM_DOES_NOT_EXIST_MESSAGE),
                Response::HTTP_NOT_FOUND
            );
        }

        return $user;
    }

    public function getCount(array $filterData = []): int
    {
        $filter = $this->getFilterData($filterData);

        $query = $this->getQuery();

        if (! $filter['with_deleted']) {
            $query->whereNull("{$this->table}.deleted_at");
        }

        return $query->count();
    }

    /**
     * @throws Exception
     */
    public function create(array $data): object
    {
        try {
            $data = $this->prepareForDB($data);
            $userId = $this->getQuery()->insertGetId($data);
            $user = MobileAppSection::find($userId);

            return $user;
        } catch (Exception $exception) {
            throw new Exception($exception->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * @throws Exception
     */
    public function prepareForDB(array $data, ?object $item = null): array
    {
        $data = parent::prepareForDB($data, $item);
        if (empty($item)) {
            $data['created_at'] = now();
            $data['institute_id'] = $this->getCurrentInstituteId();
            $data['branch_id'] = $this->getCurrentBranchId();
            if (! empty($data['image']) && $data['image'] instanceof \Illuminate\Http\UploadedFile) {
                $data['image'] = fileUploader('mobile_app_sections/', 'png', $data['image']);
            }
        } else {
            if (! empty($data['image']) && $data['image'] instanceof \Illuminate\Http\UploadedFile) {
                $data['image'] = fileUploader('mobile_app_sections/', 'png', $data['image'], $item->image);
            }
            $data['updated_at'] = now();
        }

        return $data;
    }

    /**
     * @throws Exception
     */
    public function update(int $id, array $data): ?object
    {
        try {
            $user = MobileAppSection::find($id);
            $data = $this->prepareForDB($data, $user);
            parent::update($id, $data);

            return $this->getById($id);
        } catch (Exception $exception) {
            throw new Exception($exception->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    protected function getExceptionMessages(): array
    {
        $exceptionMessages = parent::getExceptionMessages();

        $userExceptionMessages = [
            static::MESSAGE_ITEM_DOES_NOT_EXIST_MESSAGE => 'MobileAppSection does not exist.',
            static::MESSAGE_ITEM_COULD_NOT_BE_DELETED => 'MobileAppSection could not be deleted.',
        ];

        return array_merge($exceptionMessages, $userExceptionMessages);
    }
}
